function [d_stks,psf] = deconv_vstk2(varargin)
%little function that applies a 2d deconvolution to a virtual stack of images.
%Synatax:   [d_stk,psf] = deconv_vstk('edgetaper',1,'deconv','reg','numit',10,...
%               'initpsf',10);
%Input:     'edgetaper' = taper the image edges according to the PSF.
%               Default = 0; off.
%           'deconv' = type of deconvolution used.  Default = lucy
%               Lucy-Richardson.  Other types: 'reg' = regularized,
%               'wiener' = wiener, and 'blind' = blind, based on maximum
%               likelihood algorithm.
%           'numit' = number of iteration for deconvolution, default = 10.
%               only applicable to blind and Lucy-Richardson.
%           'dampar' = the threshold deviation of the resulting image from
%               the original image (in terms of the standard deviation of
%               Poisson noise) below which damping occurs
%           'subsmpl' =  denotes subsampling and is used when the PSF is
%               given on a grid that is SUBSMPL times finer than the image.
%               Default is 1.
%           'initpsf' = the size of the initial psf generated for the blind
%               deconvolution.  The PSF restoration is affected strongly by
%               the size of the initial guess INITPSF and less by the
%               values it contains.
%           'cfx' = change file data types to preserve flux ratios. Default
%               = 0, off.  In this case, a 8 bit image will go to 16 and
%               back to 8bit, 16 bit image will go to 32bit then back to 16
%               for storage.
%           'save' = save the stacks as tiff files automatically. Default =
%               1.  0 = off.
%           'multi' = multithreading stacks.  Default = 1 or ON.
%Output:    d_stk = the deconvoluted image stack
%           psf = the PSF used.
%           fac = the psf parameter used. (only if lucy is used)

[initpsf,edgetapered,deconv_type,numit,dampar,subsmpl,sav,cfx,multi] = parse(varargin);

%localization for macs and linux
if ispc
    slash = '\';        %Windows directory marker
else
    slash = '/';        %Mac directory marker
end

%now lets get the directories with the files
prompt_box('title','Select Image Directory','prompt1','Open the root folder of your images','position','center');
pause(0.25);
dir_root = uigetdir2('','Directory where the images are located');    %get the directory

%now get the directory of the PSF files
prompt_box('title','Select PSF Directory','prompt1','Open the folder of your PSFs','position','center');
pause(0.25);
psf_dir = uigetdir3('','Directory where the PSFs are located');    %get the directory

%get the psf filenames and parse it first
psf_files = dir(psf_dir);
%truncate the first two rows which are . and ..
psf_files = {psf_files.name};
for i = 3:size(psf_files,2)
    psf_wave(i) = psf_files{i}(1:end-4);  %don't want .tif or anything like that
end

%now grab look in the root directory to see how many directories are there
curr_dir = dir(dir_root);
%Here things will bifricate, if the root has directories then all the files
%in the root will be ignored, but if there are no directories in the root,
%then this will go into file deconvolution mode.
dirfile = [curr_dir.isdir];   %grab all of the isdir numbers
dirnames = {curr_dir.name};   %grab the all of the names in the root
filenames = dirnames(~dirfile);     %grab the file names
dirnames = dirnames(dirfile);   %now we only want the directory names, 
dirnames = dirnames(3:end);     %without the first two which are . and ..
dirfile = max(dirfile(3:end));    %create the switch

%create the root deconv directory
mkdir(dir_root,'deconv');

switch dirfile
    case 1  %directory mode
        %lets check if the root directory have a session metadata file
        isrootit = max(strcmp(filenames,'session_metadata.txt'));    %is the root the final directory
        if isrootit
            iterate = 1;    %if this is it, then do deconv only for this directory
            dirnames = {dir_root};  %set dirnames to dir_root
            dir_root = [];          %clear dir_root
        else    %we need to dig deeper
            iterate = size(dirnames,2);
        end
        for l = 1:iterate
            dir_tmp = [dir_root,slash,dirnames{l}];
            try     %see if this directory has a session metadata file, if not skip
                %grab the session_metadata.txt
                fid = fopen([dir_tmp,slash,'session_metadata.txt']);  %grab the file ID
                data_tmp = textscan(fid,'%s %d16 %s','Delimiter','\t','Headerlines',3);
                wavelength = data_tmp{3};   %grab the wavelength/filter cube info
                ch_names = data_tmp{1};     %grab the channel names/directory names.
                %parse the directory structure to see which PSF to apply
                for j = 1:size(ch_names,2)    %go through the directories and files, except the first two again
                    %not lets see if there is a match
                    try
                        decon_wave = strcmp(psf_wave,wavelength(j));
                    catch you  %no numbers in the directory name
                        msg = you.message;  %catch the error
                        decon_wave = 0;
                    end
                    %PSF is needed
                    if ~strcmp(deconv_type,'blind')          %if the deconvolution is not blind open PSF
                        if max(decon_wave)    %there is a match go go go
                            %lets open the psf
                            psf = imread([psf_dir,slash,psf_files{decon_wave==1}]);
                        else    %no match kick out to user
                            prompt_box('title','Import the PSF','prompt1','Select PSF for folder:',...
                                'prompt2',ch_names{j},'position','center');
                            pause(0.25)
                            [psf_filename,psf_path] = uigetfile2b({'*.tif','Tif files (*.tif)';'*.tiff','Tiff Files (*.tiff)';...
                                '*.*','All Files';},'Open PSF','Multiselect','off');
                            psf = imread([psf_path,psf_filename]);
                        end
                        psf = im2double(psf);       %the PSF needs to be in double percision.
                        %matlab expect the PSF to sum to 1, thus lets do a
                        %little math to make sure
                        psf = psf./(sum(sum(psf)));
                    else                                %if the deconvolution is blind generate PSF
                        psf = fspecial('gaussian',initpsf,10);
                    end
                    %now grab all of the files in that directory
                    curr_dir = dir([dir_tmp,slash,ch_names{j}]);  %now get the directory structure of the next level
                    %truncate the first two rows which are . and ..
                    file_tmp = {curr_dir.name};     %grab the names
                    file_tmp = file_tmp(~[curr_dir.isdir]);     %only the filenames are taken
                    %create the channel directory in deconv root
                    mkdir([dir_tmp,slash,'deconv'],ch_names{j});
                    for k = 1:size(file_tmp,2)  %go through the files
                        curr_img = imread([dir_tmp,slash,ch_names{j},slash,file_tmp{k}]);
                        %process the image info
                        imgclass = class(curr_img);      %get image type
                        switch imgclass
                            case 'uint8'
                                dampar = uint8(dampar);      %the image is unsigned 8bit
                                if cfx      %preserve flux
                                    curr_img = uint16(curr_img);
                                    imgclass = 'uint16';
                                    dampar = uint16(dampar);
                                end
                            case 'uint16'
                                dampar = uint16(dampar);     %the image is unsigned 16bit
                                if cfx
                                    curr_img = im2double2(uint32(curr_img));
                                    imgclass = 'uint32b';
                                    dampar = double(dampar);
                                end
                            case 'uint32'                    %no support for 32bit yet, convert to double
                                %dampar = uint32(dampar);
                                curr_img = im2double2(curr_img);
                                dampar = double(dampar);
                            case 'double'
                                dampar = double(dampar);     %the image is double
                            case 'single'
                                dampar = single(dampar);     %the image is single
                        end
                        %now let the fun begin
                        if edgetapered        %if the edges need to be tapered
                            curr_img = edgetaper(curr_img,psf);
                        end
                        %now apply the deconvolution
                        switch deconv_type
                            case 'lucy'
                                if multi    %if going for multi threads
                                    tic
                                    curr_img = deconvlucy(curr_img,psf,numit,dampar,[],[],subsmpl);
                                    toc
                                else        %single thread only
                                    curr_img = deconvlucy(curr_img,psf,numit,dampar,[],[],subsmpl);
                                end
                            case 'reg'
                                tic
                                curr_img = deconvreg(curr_img,psf);
                                toc
                            case 'wiener'
                                tic
                                curr_img = deconvwnr(curr_img,psf);
                                toc
                            case 'blind'
                                initpsf = psf;      %psf is the output so it will change, temp the PSF.
                                tic
                                [curr_img,psf] = deconvblind(curr_img,initpsf,numit,dampar);
                                toc
                        end
                        %stretch dynamic range (decomment if you want it on)
                        %stk_out = imnorm(stk_out);
                        %make the cfx output folder if necessary
                        if cfx&&strcmp('uint32b',imgclass)&&k==1
                            mkdir([dir_tmp,slash,'deconv'],[ch_names{j},'_16bit']);
                        end
                        %if cfx is on change the data back to the initial data type
                        if cfx
                            switch imgclass
                                case 'uint32'                    %32bit in 32bit out
                                    curr_img = uint32(curr_img.*4294967295);
                                case 'uint32b'                    %16bit in 16bit out a special case I guess
                                    curr_img2 = im2uint16(imnorm(curr_img));          %output the data in 16bit for compatibility and ease of use
                                    curr_img = uint32(curr_img.*4294967295);      %output the full image
                                otherwise
                                    curr_img = im2uint16(curr_img);   %uint16 is default
                            end
                        else  %otherwise make sure the output is uint16 or unint32 (comment out if you do not want this)
                            switch imgclass
                                case 'uint32'                    %32bit in 32bit out
                                    curr_img = uint32(curr_img.*4294967295);
                                otherwise
                                    curr_img = im2uint16(curr_img);   %uint16 is default
                            end
                        end
                        %now output the file
                        %save the stacks automatically if selected.
                        if sav
                            stk2tiff(curr_img',file_tmp{k},[dir_tmp,slash,'deconv',slash,ch_names{j},slash]);
                            if cfx&&strcmp('uint32b',imgclass)
                                stk2tiff(curr_img2',file_tmp{k},[dir_tmp,slash,'deconv',slash,ch_names{j},'_16bit',slash]);
                            end
                        end
                    end
                end
            catch you
                msg2 = you.message;     %catch the error
            end
        end
otherwise   %file mode
    %grab the directory/file names
    file_tmp = {curr_dir.name};
    file_tmp = file_tmp(~[curr_dir.isdir]);     %only the filenames are taken
    %get the wavelength names from the root directory
    %grab only the last directory of the root
    dirmsk = isstrprop(dir_root,'punct');    %find the punctuations
    dirpunct = dir_root(dirmsk);             %grab the punctuations
    punct_idx = find(dirmsk==1);            %find the location of the punctuations
    %find the last directory demarcation
    for l = size(dirpunct,2):-1:1,
        if strcmp(dirpunct(l),slash)
            break
        end
    end
    %grab the last folder
    curr_wave = dir_root(punct_idx(l):end);
    wave_strmsk = isstrprop(curr_wave,'digit');
    %grab the wavelength from root folder
    curr_wave = str2double(curr_wave(wave_strmsk));    %grab the current wavelength
    %process PSF
    try
        decon_wave = psf_wave==curr_wave;
    catch   %no numbers in directory name
        decon_wave = 0;
    end
    %PSF is needed
    if ~strcmp(deconv_type,'blind')          %if the deconvolution is not blind open PSF
        if max(decon_wave)    %there is a match go go go
            %lets open the psf
            psf = imread([psf_dir,slash,psf_files{decon_wave==1}]);
        else    %no match kick out to user
            prompt_box('title','Import the PSF','prompt1','Select PSF for folder:',...
                'prompt2',dir_root(punct_idx(l):end),'position','center');
            pause(0.25)
            [psf_filename,psf_path] = uigetfile2b({'*.tif','Tif files (*.tif)';'*.tiff','Tiff Files (*.tiff)';...
                '*.*','All Files';},'Open PSF','Multiselect','off');
            psf = imread([psf_path,psf_filename]);
        end
        psf = im2double(psf);       %the PSF needs to be in double percision.
        %matlab expect the PSF to sum to 1, thus lets do a
        %little math to make sure
        psf = psf./(sum(sum(psf)));
    else                                %if the deconvolution is blind generate PSF
        psf = fspecial('gaussian',initpsf,10);
    end
    %now go through the files
    for j = 1:size(file_tmp,2)
        curr_img = imread([dir_root,slash,file_tmp{j}]);
        %process the image info
        imgclass = class(curr_img);      %get image type
        switch imgclass
            case 'uint8'
                dampar = uint8(dampar);      %the image is unsigned 8bit
                if cfx      %preserve flux
                    curr_img = uint16(curr_img);
                    imgclass = 'uint16';
                    dampar = uint16(dampar);
                end
            case 'uint16'
                dampar = uint16(dampar);     %the image is unsigned 16bit
                if cfx
                    curr_img = im2double2(uint32(curr_img));
                    imgclass = 'uint32b';
                    dampar = double(dampar);
                end
            case 'uint32'                    %no support for 32bit yet, convert to double
                %dampar = uint32(dampar);
                curr_img = im2double2(curr_img);
                dampar = double(dampar);
            case 'double'
                dampar = double(dampar);     %the image is double
            case 'single'
                dampar = single(dampar);     %the image is single
        end
        %now let the fun begin
        if edgetapered        %if the edges need to be tapered
            curr_img = edgetaper(curr_img,psf);
        end
        %now apply the deconvolution
        switch deconv_type
            case 'lucy'
                if multi    %if going for multi threads
                    tic
                    curr_img = deconvlucy(curr_img,psf,numit,dampar,[],[],subsmpl);
                    toc
                else        %single thread only
                    curr_img = deconvlucy(curr_img,psf,numit,dampar,[],[],subsmpl);
                end
            case 'reg'
                tic
                curr_img = deconvreg(curr_img,psf);
                toc
            case 'wiener'
                tic
                curr_img = deconvwnr(curr_img,psf);
                toc
            case 'blind'
                initpsf = psf;      %psf is the output so it will change, temp the PSF.
                tic
                [curr_img,psf] = deconvblind(curr_img,initpsf,numit,dampar);
                toc
        end
        %stretch dynamic range (decomment if you want it on)
        %stk_out = imnorm(stk_out);
        %create the channel directory in deconv root
        if cfx&&strcmp('uint32b',imgclass)&&j==1
            mkdir(dir_root,'deconv_16bit');
        end
        %if cfx is on change the data back to the initial data type
        if cfx
            switch imgclass
                case 'uint32'                    %32bit in 32bit out
                    curr_img = uint32(curr_img.*4294967295);
                case 'uint32b'                    %16bit in 16bit out a special case I guess
                    curr_img2 = im2uint16(imnorm(curr_img));          %output the data in 16bit for compatibility and ease of use
                    curr_img = uint32(curr_img.*4294967295);      %output the full image
                otherwise
                    curr_img = im2uint16(curr_img);   %uint16 is default
            end
        else  %otherwise make sure the output is uint16 or unint32 (comment out if you do not want this)
            switch imgclass
                case 'uint32'                    %32bit in 32bit out
                    curr_img = uint32(curr_img.*4294967295);
                otherwise
                    curr_img = im2uint16(curr_img);   %uint16 is default
            end
        end
        %now output the file
        %save the stacks automatically if selected.
        if sav
            stk2tiff(curr_img',file_tmp{j},[dir_root,slash,'deconv',slash]);
            if cfx&&strcmp('uint32b',imgclass)
                stk2tiff(curr_img2',file_tmp{j},[dir_root,slash,'deconv_16bit',slash]);
            end
        end
    end
end




%--------------------------------------------------------------------------
%subfunction to parse the inputs.
function [initpsf,edgetapered,deconv_type,numit,dampar,subsmpl,sav,cfx,multi] = parse(input)

deconv_type = 'lucy';    %1=lucy-richardson(default), 2=regularized, 3=wiener, 4=blind.
edgetapered = 0;  %off.
numit = 10;     %default # of iteration = 10.
initpsf = 10;  %default = 10x10
sav = 1;       %default = save the file automatically
dampar = 0;     %default = 0 no dampling
subsmpl = 1;    %deault = 1 no subsampling
cfx = 0;     %by default off
multi = 1;      %multi on by default

%Parse the input
if ~isempty(input)
    for i = 1:2:size(input,2)
        if ischar(input{1,i});
            switch input{1,i}
                case 'edgetaper'
                    edgetapered = input{1,i+1};
                case 'deconv'
                    if ischar(input{1,i+1})
                        deconv_type = input{1,i+1};
                    else
                        warning(['Your entered deconvolution type is not recognized, reverting to defalut type.']);
                    end
                case 'numit'
                    numit = input{1,i+1};
                case 'save'
                    sav = input{1,i+1};
                case 'dampar'
                    dampar = input{1,i+1};
                case 'subsmpl'
                    subsmpl = input{1,i+1};
                case 'initpsf'
                    initpsf = input{1,i+1};
                case 'cfx'
                    cfx = input{1,i+1};
                case 'multi'
                    multi = input{1,i+1};
                otherwise
                    warning(['Your input ',input{1,i},' is not recognized.']);
            end
        else
            error(['The parameters you entered is incorrect.  Please check help.']);
        end
    end
end